/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  FLib.h
  \brief	  This is the Flash Library Header Driver File
  \brief	  Provides functionality to animate graphics
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 47 $
  \date  	  $Date: 2010-10-06 14:50:52 -0500 (Wed, 06 Oct 2010) $  
  
  * History:  5/Jan/2009  - Initial Version
  			  2/Sept/2010 - Changed FLIB_AnimSetupCustom for a different prototype for future support of Vector Graphics
  			  30/Sept/2010 - FLIB_AnimCancel added.
  			  6/Oct/2010 - Updated custom animation: functionality OK 		

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/


#ifndef 	_FLIB_H_
#define  	_FLIB_H_	 

#include "..\HAL\DMA_drv.h"
#include "..\HAL\DCU_drv.h"
#include "..\HAL\GALLO.h"
#include	 "Graphics.h"

/******************************* Data Types **********************************/

/* Position structure */
typedef  struct
{
	int16_t  x;	/**< the x coordinate */
	int16_t  y;	/**< the y coordinate */
} FLIB_Position_t;

/* Color structure */
typedef  struct
{
    uint8_t alpha;	/**< the alpha channel */
    uint8_t red;	/**< the red channel */
    uint8_t green;	/**< the green channel */
    uint8_t blue;	/**< the blue channel */    
} FLIB_Color_t;

/* Interpolation type enumeration */
typedef  enum
{
	FLIB_LINEAR,	/**< Represents linear interpolation (constant change) */
	FLIB_DYNAMIC,	/**< Represents dynamic interpolation, first an accel ramp, steady speed and finally deaccel ramp */
	FLIB_TABLE	/**< Represents custom interpolation, uses a table to determine change */
} FLIB_Interpolation_t;

/* Rectangle structure */
typedef  struct
{
	uint16_t  x;	/**< the x coordinate */
	uint16_t  y;	/**< the y coordinate */
	uint16_t  w;	/**< Width */
	uint16_t  h;	/**< Height */
} FLIB_Rect_t;

/* Defines a position Animation */
typedef  struct
{
	FLIB_Position_t  start;			/**< The start position */
	FLIB_Position_t  end;			/**< The ending position */
	FLIB_Interpolation_t transition;	/**< Interpolation type to use */
	const FLIB_Position_t* inter_Table;	/**< Table used when Interpolation = FLIB_TABLE. Otherwise it should be NULL_PTR */
} FLIB_AnimPos_t;

/* Defines an alpha Animation */
typedef  struct
{
	uint8_t  start;				/**< The start alpha value */
	uint8_t  end; 				/**< The ending alpha value */
	FLIB_Interpolation_t transition;	/**< Interpolation type to use */
	const uint8_t* inter_Table;		/**< Table used when Interpolation = FLIB_TABLE. Otherwise it should be NULL_PTR */
} FLIB_AnimAlpha_t;

/* Defines a color Animation */
/* Each color array, must have two colors (one for FG and one for BG)*/
typedef  struct
{
    const FLIB_Color_t*	start;			/**< The starting color array (The array MUST contain two elements) */
    const FLIB_Color_t*	end; 			/**< The ending color array (The array MUST contain two elements) */
} FLIB_AnimColor_t;

/* Defines a scroll Animation */
typedef  struct
{
	FLIB_Rect_t      start;			/**< The starting bounds */
	FLIB_Position_t  end;			/**< The ending position */
	FLIB_Interpolation_t transition;	/**< Interpolation type to use */
	const FLIB_Position_t* inter_Table;	/**< Table used when Interpolation = FLIB_TABLE. Otherwise it should be NULL_PTR */
} FLIB_AnimScrolling_t;

/* Defines a scaling Animation (Integers) */
typedef  struct
{
	uint16_t  start;			/**< The start scale value */
	uint16_t  end; 				/**< The ending scale value */
	FLIB_Interpolation_t transition;	/**< Interpolation type to use */
	const uint16_t* inter_Table;		/**< Table used when Interpolation = FLIB_TABLE. Otherwise it should be NULL_PTR */
} FLIB_AnimScaling_t;

/* Defines an Object Animation */
typedef  struct
{
    const uint16_t* index;			/**< It is an array of indices, for the object list */    
    const Graphics_Object_t* *objlist;		/**< It is an array of objects */  
} FLIB_AnimObject_t;

/* Defines a size Animation */
typedef  struct
{
	FLIB_Rect_t	start;					/**< The starting bounds */
	FLIB_Rect_t end;					/**< The ending bounds */
	FLIB_Interpolation_t transition;	/**< Interpolation type to use */
	const FLIB_Rect_t* inter_Table;		/**< Table used when Interpolation = FLIB_TABLE. Otherwise it should be NULL_PTR */
} FLIB_AnimSize_t;

/* Enumeration of the current state of the state machine */
typedef  enum
{
	FLIB_STATE_OK,			/**< State machine ready to process the next frame */
	FLIB_STATE_UPDATE_PENDING,	/**< State machine is still processing the current frame */
	FLIB_STATE_UNINIT		/**< The library has not been initialized */
} FLIB_State_t; 

/* Error return cases */
typedef  enum
{
	FLIB_ERROR_OK,			/**< Succesful operation */
	FLIB_ERROR_NO_CALLBACK,		/**< The state machine is missing the callback configuration */
	FLIB_ERROR_LAYER_ACTIVE,	/**< The selected layer is already being used */

	FLIB_ERROR_ANIM_POS,		/**< Error in position data */
	FLIB_ERROR_ANIM_ALPHA,		/**< Error in alpha data */
	FLIB_ERROR_ANIM_COLOR,		/**< Error in color data */
	FLIB_ERROR_ANIM_SCROLLING,	/**< Error in scroll data */
	FLIB_ERROR_ANIM_OBJECT,		/**< Error in object data */
	FLIB_ERROR_ANIM_SIZE,		/**< Error in object data */

	FLIB_ERROR_DMA,				/**< DMA related error */
	FLIB_ERROR_UNINIT,			/**< Driver uninit related error */
	
	FLIB_ERROR_PARAM,			/**< wrong parameters related error */
	FLIB_ERROR_FULL				/**< Not enough space */
} FLIB_Error_t;


typedef  void  ( * FLIB_CallbackType )( uint32_t frameNumber );
typedef  void  ( * FLIB_CallbackAnim_t )( uint32_t frameNumber, void *aniData);

#define FLIB_FOREVER	(0xFFFFFFFF)

/************************* Configuration Parameters **************************/

/* Requires number of layer x entities */
#define FLIB_MEMSTARTNUMBER	(64)
#define FLIB_MAXANICTX0		(64)
#define FLIB_MAXANICTX1		(4)
#define FLIB_MAXANICUSTOM	(64)
#define FLIB_FPS			(60)
#define FLIB_SLOTS			(64)

/***************************** Public Functions ******************************/
void FLIB_SetContextDCU(void);
#ifdef DCULITE	
void FLIB_SetContextDCULITE(void);
#endif 
void FLIB_Init(void);
void FLIB_DeInit(void);
void FLIB_AnimCancel(uint8_t Layer);
void FLIB_Setup(FLIB_CallbackType Callback);
FLIB_Error_t FLIB_RenderNextFrame(void);
FLIB_State_t FLIB_GetState(void);

/* Animation Setup */
FLIB_Error_t FLIB_AnimSetup
( 
				 uint8_t Layer,
	             uint32_t duration,
			     const Graphics_Object_t *gobject,
			     const FLIB_AnimPos_t* pos,
			     const FLIB_AnimAlpha_t* alpha,
			     const FLIB_AnimColor_t* color,
			     const FLIB_AnimScrolling_t* scrolling,
			     const FLIB_AnimScaling_t* scaling,
			     const FLIB_AnimObject_t* object
);

FLIB_Error_t FLIB_AnimSetupPos
(
	uint8_t Layer,
	uint32_t duration,
	const Graphics_Object_t *gobject,	
	const FLIB_AnimPos_t* pos				
);

FLIB_Error_t FLIB_AnimSetupAlpha
(
	uint8_t Layer,
	uint32_t duration,
	const Graphics_Object_t *gobject,
	const FLIB_AnimAlpha_t* alpha				
);

FLIB_Error_t FLIB_AnimSetupSize
(
	uint8_t Layer,
	uint32_t duration,
	const Graphics_Object_t *gobject,
	const FLIB_AnimSize_t* size,
	const FLIB_AnimAlpha_t* alpha,
	const FLIB_AnimColor_t* color
);

FLIB_Error_t FLIB_AnimSetupCustom
(
	uint32_t	duration, 			//	duration or FLIB_FOREVER
	uint8_t		rate,				//  repetition rate
	uint8_t		offset,				//	offset, based on the rate. Range: [0,rate - 1]
	FLIB_CallbackType callback		//	callback, arguments contain relative frame, duration...
);


void FLIB_Sync(void);


#endif
